import React, { useState } from 'react';
import { Assistant } from '../types/assistant';
import { Phone, Loader2, MessageSquare } from 'lucide-react';
import { formatPhoneNumber, isValidPhoneNumber } from '../utils/phoneValidation';
import { scheduleCall } from '../lib/vapi';

interface AssistantCardProps {
  assistant: Assistant;
}

export default function AssistantCard({ assistant }: AssistantCardProps) {
  const [showForm, setShowForm] = useState(false);
  const [showGreeting, setShowGreeting] = useState(false);
  const [name, setName] = useState('');
  const [phoneNumber, setPhoneNumber] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  const handlePhoneChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const formatted = formatPhoneNumber(e.target.value);
    setPhoneNumber(formatted);
    setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);
    setSuccess(false);

    if (!name.trim()) {
      setError('Please enter your name');
      return;
    }

    if (!isValidPhoneNumber(phoneNumber)) {
      setError('Please enter a valid phone number');
      return;
    }

    setLoading(true);
    try {
      const cleanNumber = phoneNumber.replace(/\D/g, '');
      const response = await scheduleCall(cleanNumber, name.trim(), assistant.industry);
      
      if (response.status === 'scheduled' || response.status === 'pending') {
        setSuccess(true);
        setName('');
        setPhoneNumber('');
        setTimeout(() => {
          setShowForm(false);
          setSuccess(false);
        }, 3000);
      } else {
        throw new Error('Call scheduling failed. Please try again.');
      }
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to schedule call. Please try again.';
      setError(errorMessage);
      console.error('Error scheduling call:', err);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="group relative bg-slate-800/50 backdrop-blur-sm rounded-xl p-6 border border-slate-700/50 hover:border-indigo-400 transition-all hover:-translate-y-1 hover:shadow-xl hover:shadow-indigo-500/20 animate-slide-in">
      <div className="flex flex-col items-center text-center mb-4">
        {/* Avatar Image */}
        <div className="w-24 h-24 rounded-full overflow-hidden mb-4 border-2 border-indigo-500/30 group-hover:border-indigo-500 transition-colors">
          <img 
            src={assistant.avatar} 
            alt={assistant.name}
            className="w-full h-full object-cover"
          />
        </div>

        <h3 className="text-xl font-semibold text-white mb-1">{assistant.name}</h3>
        <p className="text-sm text-indigo-300">{assistant.title}</p>
        
        <button
          onClick={() => setShowGreeting(!showGreeting)}
          className="mt-2 text-sm text-slate-400 hover:text-indigo-300 flex items-center gap-1 transition-colors"
        >
          <MessageSquare className="h-4 w-4" />
          <span>{showGreeting ? 'Hide greeting' : 'View greeting'}</span>
        </button>
        
        {showGreeting && (
          <div className="mt-3 p-3 bg-slate-900/50 rounded-lg text-sm text-slate-300 animate-slide-in border border-indigo-500/20">
            {assistant.greeting}
          </div>
        )}
        
        <p className="text-sm text-slate-400 mt-2">{assistant.description}</p>
      </div>

      {success && (
        <div className="mb-4 p-3 bg-green-500/30 border border-green-400/40 text-green-300 rounded-lg text-center animate-slide-in">
          Call scheduled successfully! We'll call you shortly.
        </div>
      )}

      {error && (
        <div className="mb-4 p-3 bg-red-500/30 border border-red-400/40 text-red-300 rounded-lg text-center animate-slide-in">
          {error}
        </div>
      )}

      {!showForm ? (
        <button
          onClick={() => setShowForm(true)}
          className="w-full py-3 px-4 bg-indigo-600/30 hover:bg-indigo-600/40 text-indigo-300 rounded-lg transition-colors flex items-center justify-center space-x-2 group-hover:text-white animate-pulse-ring border border-indigo-500/30"
        >
          <Phone className="h-4 w-4" />
          <span>Contact {assistant.name}</span>
        </button>
      ) : (
        <form onSubmit={handleSubmit} className="space-y-4 animate-slide-in">
          <input
            type="text"
            placeholder="Your Name"
            value={name}
            onChange={(e) => setName(e.target.value)}
            className="w-full px-4 py-2 bg-slate-900/50 border border-indigo-500/30 rounded-lg text-white placeholder-slate-400 focus:border-indigo-400 focus:ring-1 focus:ring-indigo-400 transition-colors"
            disabled={loading}
          />
          <div className="relative">
            <input
              type="tel"
              placeholder="+1 (415) 123-4567"
              value={phoneNumber}
              onChange={handlePhoneChange}
              className="w-full px-4 py-2 bg-slate-900/50 border border-indigo-500/30 rounded-lg text-white placeholder-slate-400 focus:border-indigo-400 focus:ring-1 focus:ring-indigo-400 transition-colors"
              disabled={loading}
            />
          </div>
          <button
            type="submit"
            disabled={loading}
            className="w-full py-2 px-4 bg-indigo-600 hover:bg-indigo-500 text-white rounded-lg transition-colors flex items-center justify-center space-x-2 disabled:opacity-50 animate-glow"
          >
            {loading ? (
              <Loader2 className="h-4 w-4 animate-spin" />
            ) : (
              <Phone className="h-4 w-4" />
            )}
            <span>{loading ? 'Scheduling...' : `Call ${assistant.name}`}</span>
          </button>
        </form>
      )}

      <div className="absolute top-3 right-3 flex space-x-1">
        <div className="px-2 py-1 rounded-full bg-slate-900/50 border border-indigo-500/30">
          <span className="text-xs text-indigo-300">{assistant.stats.activeSessions} active</span>
        </div>
        <div className="px-2 py-1 rounded-full bg-slate-900/50 border border-indigo-500/30">
          <span className="text-xs text-green-400">{assistant.stats.successRate}%</span>
        </div>
      </div>
    </div>
  );
}