import React, { useState } from 'react';
import { DollarSign, Loader2, MessageSquare } from 'lucide-react';

export default function SalesAssistant() {
  const [email, setEmail] = useState('');
  const [topic, setTopic] = useState('pricing');
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const topics = [
    { id: 'pricing', label: 'Pricing & Plans' },
    { id: 'enterprise', label: 'Enterprise Solutions' },
    { id: 'integration', label: 'API Integration' },
    { id: 'custom', label: 'Custom Requirements' }
  ];

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);
    setSuccess(false);
    setLoading(true);

    try {
      await new Promise(resolve => setTimeout(resolve, 1500));
      setSuccess(true);
      setEmail('');
      setTopic('pricing');
    } catch (err) {
      setError('Failed to schedule consultation. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="relative bg-slate-900/50 backdrop-blur-xl rounded-3xl border border-emerald-500/20 overflow-hidden group">
      {/* Animated Background */}
      <div className="absolute inset-0 bg-gradient-to-br from-emerald-500/10 via-teal-500/10 to-cyan-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
      <div className="absolute inset-0 bg-gradient-radial from-emerald-500/20 via-transparent to-transparent animate-pulse-slow"></div>

      <div className="relative p-8">
        <div className="flex items-center gap-4 mb-8">
          <div className="w-12 h-12 rounded-2xl bg-gradient-to-br from-emerald-500 to-teal-600 flex items-center justify-center animate-pulse-ring">
            <DollarSign className="h-6 w-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold text-white">Sales Consultation</h2>
            <p className="text-emerald-300">Discover the perfect solution for your business</p>
          </div>
        </div>

        {success && (
          <div className="mb-6 p-4 bg-emerald-500/20 border border-emerald-500/30 text-emerald-300 rounded-2xl animate-slide-in">
            Consultation scheduled! We'll contact you shortly.
          </div>
        )}

        {error && (
          <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-300 rounded-2xl animate-slide-in">
            {error}
          </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-6">
          <div>
            <label className="block text-sm font-medium text-emerald-300 mb-2">Business Email</label>
            <div className="relative">
              <MessageSquare className="absolute left-4 top-1/2 transform -translate-y-1/2 h-5 w-5 text-emerald-400" />
              <input
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                className="w-full pl-12 pr-4 py-3 bg-slate-800/50 border border-emerald-500/30 rounded-xl 
                         text-white placeholder-emerald-300/50 focus:border-emerald-400 focus:ring-1 
                         focus:ring-emerald-400 transition-all"
                placeholder="you@company.com"
                required
                disabled={loading}
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium text-emerald-300 mb-2">Discussion Topic</label>
            <select
              value={topic}
              onChange={(e) => setTopic(e.target.value)}
              className="w-full pl-4 pr-10 py-3 bg-slate-800/50 border border-emerald-500/30 rounded-xl 
                       text-white focus:border-emerald-400 focus:ring-1 focus:ring-emerald-400 transition-all"
              disabled={loading}
            >
              {topics.map(t => (
                <option key={t.id} value={t.id} className="bg-slate-800">
                  {t.label}
                </option>
              ))}
            </select>
          </div>

          <button
            type="submit"
            disabled={loading}
            className="w-full py-4 px-6 bg-gradient-to-r from-emerald-600 to-teal-600 text-white rounded-xl 
                     font-medium flex items-center justify-center space-x-2 disabled:opacity-50 
                     hover:from-emerald-500 hover:to-teal-500 transition-all hover:scale-105 animate-glow"
          >
            {loading ? (
              <Loader2 className="h-5 w-5 animate-spin" />
            ) : (
              <MessageSquare className="h-5 w-5" />
            )}
            <span>{loading ? 'Scheduling...' : 'Schedule Consultation'}</span>
          </button>
        </form>
      </div>
    </div>
  );
}