import React, { useState } from 'react';
import { Phone, Clock, Loader2, Bot } from 'lucide-react';
import { scheduleCall } from '../lib/vapi';
import { formatPhoneNumber, isValidPhoneNumber } from '../utils/phoneValidation';

export default function CallScheduler() {
  const [phoneNumber, setPhoneNumber] = useState('');
  const [name, setName] = useState('');
  const [scheduleTime, setScheduleTime] = useState('10:00');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  const handlePhoneChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const formatted = formatPhoneNumber(e.target.value);
    setPhoneNumber(formatted);
    setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);
    setSuccess(false);

    if (!name.trim()) {
      setError('Please enter your name');
      return;
    }

    if (!isValidPhoneNumber(phoneNumber)) {
      setError('Please enter a valid phone number');
      return;
    }

    setLoading(true);
    try {
      const result = await scheduleCall(phoneNumber.replace(/\D/g, ''), name.trim());
      
      if (result.status === 'scheduled') {
        setSuccess(true);
        setPhoneNumber('');
        setName('');
        setScheduleTime('10:00');
      } else {
        throw new Error(result.error || 'Failed to schedule call');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to schedule call');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="relative bg-slate-900/50 backdrop-blur-xl rounded-3xl border border-indigo-500/20 overflow-hidden group">
      <div className="absolute inset-0 bg-gradient-to-br from-indigo-500/10 via-purple-500/10 to-blue-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
      <div className="absolute inset-0 bg-gradient-radial from-indigo-500/20 via-transparent to-transparent animate-pulse-slow"></div>

      <div className="relative p-8">
        <div className="flex items-center gap-4 mb-8">
          <div className="w-12 h-12 rounded-2xl bg-gradient-to-br from-indigo-500 to-purple-600 flex items-center justify-center animate-pulse-ring">
            <Bot className="h-6 w-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold text-white">Schedule Your Live AI Demo</h2>
            <p className="text-indigo-300">Experience the future of AI communication</p>
          </div>
        </div>

        {success && (
          <div className="mb-6 p-4 bg-emerald-500/20 border border-emerald-500/30 text-emerald-300 rounded-2xl animate-slide-in">
            <p className="flex items-center gap-2">
              <Clock className="h-5 w-5" />
              Demo call scheduled successfully! We'll call you at the scheduled time.
            </p>
          </div>
        )}

        {error && (
          <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-300 rounded-2xl animate-slide-in">
            {error}
          </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-6">
          <div>
            <label className="block text-sm font-medium text-indigo-300 mb-2">Your Name</label>
            <input
              type="text"
              value={name}
              onChange={(e) => setName(e.target.value)}
              className="w-full px-4 py-3 bg-slate-800/50 border border-indigo-500/30 rounded-xl 
                       text-white placeholder-indigo-300/50 focus:border-indigo-400 focus:ring-1 
                       focus:ring-indigo-400 transition-all"
              placeholder="Enter your name"
              disabled={loading}
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-indigo-300 mb-2">Phone Number</label>
            <div className="relative">
              <Phone className="absolute left-4 top-1/2 transform -translate-y-1/2 h-5 w-5 text-indigo-400" />
              <input
                type="tel"
                value={phoneNumber}
                onChange={handlePhoneChange}
                className="w-full pl-12 pr-4 py-3 bg-slate-800/50 border border-indigo-500/30 rounded-xl 
                         text-white placeholder-indigo-300/50 focus:border-indigo-400 focus:ring-1 
                         focus:ring-indigo-400 transition-all"
                placeholder="+1 (555) 000-0000"
                disabled={loading}
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium text-indigo-300 mb-2">Preferred Demo Time</label>
            <div className="relative">
              <Clock className="absolute left-4 top-1/2 transform -translate-y-1/2 h-5 w-5 text-indigo-400" />
              <input
                type="time"
                value={scheduleTime}
                onChange={(e) => setScheduleTime(e.target.value)}
                className="w-full pl-12 pr-4 py-3 bg-slate-800/50 border border-indigo-500/30 rounded-xl 
                         text-white focus:border-indigo-400 focus:ring-1 focus:ring-indigo-400 transition-all"
                disabled={loading}
              />
            </div>
          </div>

          <button
            type="submit"
            disabled={loading}
            className="w-full py-4 px-6 bg-gradient-to-r from-indigo-600 to-purple-600 text-white rounded-xl 
                     font-medium flex items-center justify-center space-x-2 disabled:opacity-50 
                     hover:from-indigo-500 hover:to-purple-500 transition-all hover:scale-105 animate-glow"
          >
            {loading ? (
              <Loader2 className="h-5 w-5 animate-spin" />
            ) : (
              <Phone className="h-5 w-5" />
            )}
            <span>{loading ? 'Scheduling...' : 'Schedule Your Demo'}</span>
          </button>
        </form>
      </div>
    </div>
  );
}