import React, { useState } from 'react';
import { ChevronDown, Bot, Search } from 'lucide-react';

interface FAQItem {
  question: string;
  answer: string;
  category?: string;
}

const faqs: FAQItem[] = [
  {
    question: "What is ClaraAI and how does it work?",
    answer: "ClaraAI is an advanced AI-powered voice assistant platform that uses natural language processing and machine learning to handle customer interactions. It creates human-like conversations through customizable voice agents that can handle calls, schedule appointments, and provide customer support.",
    category: "General"
  },
  {
    question: "How accurate is ClaraAI's voice recognition?",
    answer: "ClaraAI's voice recognition system achieves over 98% accuracy across multiple languages and accents. Our system continuously learns and improves through machine learning, adapting to different speaking patterns and industry-specific terminology.",
    category: "Technical"
  },
  {
    question: "What languages does ClaraAI support?",
    answer: "ClaraAI currently supports over 30 languages including English, Spanish, French, German, Italian, Portuguese, Japanese, Korean, Mandarin Chinese, and Arabic. Each language comes with multiple accent variations and regional dialects.",
    category: "Features"
  },
  {
    question: "How secure is my data with ClaraAI?",
    answer: "We implement enterprise-grade security measures including end-to-end encryption, secure data centers, and regular security audits. We're compliant with GDPR, HIPAA, and other major data protection regulations. Your data is encrypted both in transit and at rest.",
    category: "Security"
  },
  {
    question: "Can I customize the voice and personality of my AI assistant?",
    answer: "Yes! ClaraAI offers extensive customization options. You can select from various voice types, adjust speaking pace and tone, and customize the personality to match your brand. You can also create custom scripts and responses for specific scenarios.",
    category: "Features"
  },
  {
    question: "What industries does ClaraAI serve?",
    answer: "ClaraAI serves a wide range of industries including healthcare, real estate, financial services, retail, technology, hospitality, and more. Our AI assistants can be customized for specific industry terminology and requirements.",
    category: "General"
  },
  {
    question: "How does pricing work?",
    answer: "Our pricing is based on usage minutes and features needed. We offer flexible plans starting from $44/month for small businesses up to custom enterprise solutions. All plans include basic features, with premium features available in higher tiers.",
    category: "Pricing"
  },
  {
    question: "Do you offer a trial period?",
    answer: "Yes, we offer a 14-day free trial with full access to all features. This allows you to test ClaraAI's capabilities and customize it to your needs before committing to a subscription.",
    category: "Pricing"
  },
  {
    question: "How long does it take to set up ClaraAI?",
    answer: "Basic setup can be completed in under 30 minutes. For custom implementations, our team provides dedicated support to ensure smooth integration, typically completed within 1-2 business days.",
    category: "Technical"
  },
  {
    question: "Can ClaraAI integrate with my existing systems?",
    answer: "Yes, ClaraAI offers robust API integration capabilities and can integrate with most CRM systems, scheduling software, and business tools including Salesforce, HubSpot, Zoom, Google Calendar, and more.",
    category: "Technical"
  },
  {
    question: "What kind of support do you offer?",
    answer: "We provide 24/7 technical support through multiple channels including email, phone, and chat. Enterprise plans include dedicated account managers and priority support response times.",
    category: "Support"
  },
  {
    question: "How does ClaraAI handle complex conversations?",
    answer: "ClaraAI uses advanced natural language processing to understand context, handle multiple topics in a single conversation, and maintain coherent dialogue flow. It can recognize and respond to emotional cues and adjust its communication style accordingly.",
    category: "Technical"
  }
];

export default function FAQ() {
  const [searchTerm, setSearchTerm] = useState('');
  const [openIndex, setOpenIndex] = useState<number | null>(null);

  const filteredFaqs = faqs.filter(faq =>
    faq.question.toLowerCase().includes(searchTerm.toLowerCase()) ||
    faq.answer.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="py-24 relative overflow-hidden">
      {/* Background Effects */}
      <div className="absolute inset-0 bg-gradient-to-b from-slate-950 via-indigo-950/20 to-slate-950"></div>
      <div className="absolute inset-0 bg-[radial-gradient(ellipse_at_center,rgba(99,102,241,0.1),transparent_50%)]"></div>

      <div className="relative max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-16">
          <h2 className="text-4xl font-bold mb-4">
            <span className="text-white">Frequently Asked </span>
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-blue-400 to-indigo-400">
              Questions
            </span>
          </h2>
          <p className="text-lg text-blue-200 max-w-2xl mx-auto">
            Find answers to common questions about ClaraAI
          </p>
        </div>

        {/* Search Bar */}
        <div className="mb-12">
          <div className="relative">
            <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
            <input
              type="text"
              placeholder="Search questions..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full pl-12 pr-4 py-3 bg-slate-900/50 border border-slate-700/50 rounded-xl 
                       text-white placeholder-slate-400 focus:border-indigo-500 focus:ring-1 
                       focus:ring-indigo-500 transition-all"
            />
          </div>
        </div>

        {/* FAQ List */}
        <div className="space-y-4">
          {filteredFaqs.map((faq, index) => (
            <div
              key={index}
              className="group bg-slate-900/50 border border-slate-700/50 rounded-xl overflow-hidden 
                       hover:border-indigo-500/30 transition-all duration-300"
            >
              <button
                onClick={() => setOpenIndex(openIndex === index ? null : index)}
                className="w-full px-6 py-4 flex items-center justify-between text-left"
              >
                <span className="text-lg font-medium text-white">{faq.question}</span>
                <ChevronDown
                  className={`w-5 h-5 text-slate-400 transition-transform duration-300 
                           ${openIndex === index ? 'rotate-180' : ''}`}
                />
              </button>
              
              {openIndex === index && (
                <div className="px-6 pb-4 text-slate-300 animate-slide-in">
                  {faq.answer}
                </div>
              )}
            </div>
          ))}
        </div>

        {/* Help Section */}
        <div className="mt-12 text-center">
          <p className="text-slate-400">
            Still have questions?{' '}
            <a href="#contact" className="text-indigo-400 hover:text-indigo-300 underline">
              Contact our support team
            </a>
          </p>
        </div>
      </div>
    </div>
  );
}