import React, { useState } from 'react';
import { Check, Sparkles } from 'lucide-react';

interface PricingFeature {
  text: string;
  highlight?: boolean;
}

interface PricingTier {
  name: string;
  monthlyPrice: number;
  description: string;
  features: PricingFeature[];
  highlight?: boolean;
}

const pricingTiers: PricingTier[] = [
  {
    name: "Starter",
    monthlyPrice: 44, // $37 + 18%
    description: "Perfect for small businesses getting started with AI assistance",
    features: [
      { text: "1 Custom Voice Agent" },
      { text: "500 Minutes/month" },
      { text: "Basic Analytics" },
      { text: "5 Voice Models" },
      { text: "Email Support" },
      { text: "Standard Voice Quality" },
      { text: "Basic Call Scheduling" },
      { text: "Simple Integration" }
    ]
  },
  {
    name: "Growth",
    monthlyPrice: 150, // $127 + 18%
    description: "Ideal for growing businesses ready to scale their operations",
    highlight: true,
    features: [
      { text: "5 Custom Voice Agents", highlight: true },
      { text: "2,000 Minutes/month", highlight: true },
      { text: "Advanced Analytics" },
      { text: "15 Voice Models" },
      { text: "Priority Support" },
      { text: "HD Voice Quality" },
      { text: "Advanced Call Scheduling" },
      { text: "Multiple Integrations" },
      { text: "Custom Workflows" }
    ]
  },
  {
    name: "Enterprise",
    monthlyPrice: 469, // $397 + 18%
    description: "For organizations requiring comprehensive AI solutions",
    features: [
      { text: "Unlimited Voice Agents", highlight: true },
      { text: "10,000 Minutes/month", highlight: true },
      { text: "Enterprise Analytics" },
      { text: "Unlimited Voice Models" },
      { text: "24/7 Priority Support" },
      { text: "Ultra HD Voice Quality" },
      { text: "Custom API Integration" },
      { text: "White-label Solution" },
      { text: "Dedicated Account Manager" }
    ]
  }
];

export default function Pricing() {
  const [billingCycle, setBillingCycle] = useState<'monthly' | 'annually'>('monthly');

  const getPrice = (monthlyPrice: number, cycle: 'monthly' | 'annually') => {
    if (cycle === 'annually') {
      const annualPrice = monthlyPrice * 0.85; // 15% discount
      return `$${Math.round(annualPrice)}`;
    }
    return `$${monthlyPrice}`;
  };

  return (
    <div className="py-16 relative">
      {/* Background Effects */}
      <div className="absolute inset-0 bg-gradient-to-b from-slate-950 via-indigo-950/20 to-slate-950"></div>
      <div className="absolute inset-0 bg-[radial-gradient(ellipse_at_center,rgba(99,102,241,0.15),transparent_50%)]"></div>
      
      <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-12">
          <h2 className="text-4xl font-bold text-transparent bg-clip-text bg-gradient-to-r from-white via-indigo-200 to-white mb-4">
            Choose Your Plan
          </h2>
          <p className="text-lg text-indigo-200 max-w-2xl mx-auto">
            Flexible pricing options to match your business needs. Scale your AI assistance as you grow.
          </p>
        </div>

        {/* Billing Toggle */}
        <div className="flex justify-center items-center space-x-4 mb-12">
          <span className={`text-sm ${billingCycle === 'monthly' ? 'text-white' : 'text-indigo-400'}`}>
            Monthly
          </span>
          <button
            onClick={() => setBillingCycle(prev => prev === 'monthly' ? 'annually' : 'monthly')}
            className="relative w-14 h-7 bg-indigo-900/50 rounded-full p-1 transition-colors"
          >
            <div
              className={`w-5 h-5 rounded-full bg-indigo-500 transition-transform ${
                billingCycle === 'annually' ? 'translate-x-7' : ''
              }`}
            />
          </button>
          <div className="flex items-center space-x-1">
            <span className={`text-sm ${billingCycle === 'annually' ? 'text-white' : 'text-indigo-400'}`}>
              Annually
            </span>
            <span className="text-xs text-emerald-400 bg-emerald-900/30 px-2 py-0.5 rounded-full">
              Save 15%
            </span>
          </div>
        </div>

        {/* Pricing Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
          {pricingTiers.map((tier, index) => (
            <div
              key={index}
              className={`relative group rounded-2xl backdrop-blur-sm transition-all duration-300 hover:transform hover:-translate-y-1 ${
                tier.highlight
                  ? 'bg-gradient-to-b from-indigo-500/20 to-purple-500/20 border border-indigo-500/30'
                  : 'bg-slate-900/50 border border-slate-700/50'
              }`}
            >
              {tier.highlight && (
                <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 bg-gradient-to-r from-indigo-600 to-purple-600 text-white text-sm font-medium px-4 py-1 rounded-full flex items-center gap-1">
                  <Sparkles className="w-4 h-4" />
                  <span>Most Popular</span>
                </div>
              )}

              <div className="p-8">
                <h3 className="text-2xl font-bold text-white mb-2">{tier.name}</h3>
                <p className="text-indigo-300 mb-6">{tier.description}</p>
                
                <div className="flex items-baseline mb-8">
                  <span className="text-5xl font-bold text-white">
                    {getPrice(tier.monthlyPrice, billingCycle)}
                  </span>
                  <span className="text-indigo-300 ml-2">/month</span>
                </div>

                <button
                  className={`w-full py-3 px-6 rounded-xl mb-8 transition-all ${
                    tier.highlight
                      ? 'bg-gradient-to-r from-indigo-600 to-purple-600 text-white hover:from-indigo-500 hover:to-purple-500'
                      : 'bg-slate-800 text-indigo-300 hover:bg-slate-700'
                  }`}
                >
                  Get Started
                </button>

                <ul className="space-y-4">
                  {tier.features.map((feature, featureIndex) => (
                    <li
                      key={featureIndex}
                      className={`flex items-center space-x-3 ${
                        feature.highlight ? 'text-white' : 'text-indigo-300'
                      }`}
                    >
                      <Check className={`w-5 h-5 ${feature.highlight ? 'text-emerald-400' : 'text-indigo-500'}`} />
                      <span>{feature.text}</span>
                    </li>
                  ))}
                </ul>
              </div>

              {/* Hover Effect */}
              <div className="absolute inset-0 rounded-2xl transition-opacity opacity-0 group-hover:opacity-100">
                <div className="absolute inset-0 bg-gradient-to-r from-indigo-500/10 to-purple-500/10 animate-pulse-slow rounded-2xl"></div>
              </div>
            </div>
          ))}
        </div>

        {/* Enterprise Contact */}
        <div className="mt-12 text-center">
          <p className="text-indigo-300">
            Need a custom plan? {' '}
            <a href="#contact" className="text-indigo-400 hover:text-indigo-300 underline">
              Contact us
            </a>
          </p>
        </div>
      </div>
    </div>
  );
}