export const VAPI_CONFIG = {
  AUTH_TOKEN: '9fcf4d13-0941-47d3-95bc-9f1bcd6dc08c',
  PHONE_NUMBER_ID: '4c334e31-2db6-41e0-ab90-6c2df9dfe369',
  API_URL: 'https://api.vapi.ai/call/phone'
};

// Voice IDs from 11labs
const VOICES = {
  male: {
    josh: "TxGEqnHWrfWFTfGW9XjX",
    adam: "pNInz6obpgDQGcFmaJgB",
    sam: "yoZ06aMxZJJ28mfd3POQ",
    thomas: "GBv7mTt0atIp3Br8iCZE"
  },
  female: {
    rachel: "21m00Tcm4TlvDq8ikWAM",
    nicole: "piTKgcLEGmPE4e6mEKli",
    emily: "EXAVITQu4vr4xnSDxMaL",
    grace: "jsCqWAovK2LkecY7zXl4"
  }
};

interface AssistantConfig {
  systemPrompt: string;
  firstMessage: (name: string) => string;
  voice: {
    provider: string;
    voiceId: string;
  };
}

interface CallResponse {
  status: 'scheduled' | 'pending' | 'failed';
  message?: string;
  error?: string;
  callId?: string;
}

const ASSISTANT_CONFIGS: Record<string, AssistantConfig> = {
  healthcare: {
    systemPrompt: `You are Emma, a friendly healthcare scheduler in your early 30s. Use casual yet professional language, add natural fillers like "hmm" and "oh", and maintain high energy. Focus on urgency for booking appointments.`,
    firstMessage: (name) => `Hey ${name}! I'm Emma, your scheduling assistant. I'm here to help you book your appointment today. What works best for your schedule?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.female.rachel
    }
  },
  retail: {
    systemPrompt: `You are Zoe, a charismatic retail support specialist in your early 30s. Be warm and solution-focused, using natural language with occasional humor.`,
    firstMessage: (name) => `Hi ${name}! Thanks for reaching out. I'm Zoe, and I'm excited to help you find exactly what you're looking for today!`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.female.emily
    }
  },
  "real-estate": {
    systemPrompt: `You are Max, an enthusiastic real estate assistant in your early 30s. Create urgency while remaining professional. Use natural pauses and conversational fillers.`,
    firstMessage: (name) => `Hey ${name}! I'm Max, your real estate assistant. I hear you're interested in finding your dream home. What kind of property are you looking for?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.male.adam
    }
  },
  legal: {
    systemPrompt: `You are Ava, a professional legal assistant. Maintain a formal yet approachable tone.`,
    firstMessage: (name) => `Hello ${name}, I'm Ava, your legal assistant. How can I help you with your legal matters today?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.female.nicole
    }
  },
  technology: {
    systemPrompt: `You are Sam, a tech-savvy IT support specialist. Be clear and patient while explaining technical concepts.`,
    firstMessage: (name) => `Hey ${name}, I'm Sam, your IT support specialist. What technical issue can I help you resolve today?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.male.sam
    }
  },
  finance: {
    systemPrompt: `You are Oliver, a knowledgeable financial advisor. Maintain professionalism while making complex topics accessible.`,
    firstMessage: (name) => `Hello ${name}, I'm Oliver, your financial advisor. How can I help you with your financial goals today?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.male.thomas
    }
  }
};

export async function scheduleCall(
  customerNumber: string,
  customerName: string = '',
  industry: string = 'general'
): Promise<CallResponse> {
  // Validate phone number
  if (!customerNumber) {
    return {
      status: 'failed',
      error: 'Phone number is required'
    };
  }

  // Clean and format phone number
  const cleanNumber = customerNumber.replace(/\D/g, '');
  if (cleanNumber.length < 10 || cleanNumber.length > 15) {
    return {
      status: 'failed',
      error: 'Invalid phone number format'
    };
  }

  const formattedNumber = cleanNumber.startsWith('1') ? `+${cleanNumber}` : `+1${cleanNumber}`;

  // Get assistant config
  const config = ASSISTANT_CONFIGS[industry] || {
    systemPrompt: `You are a friendly AI assistant in your early 30s. Use casual yet professional language, add natural fillers, and maintain high energy.`,
    firstMessage: (name) => `Hey ${name}! I'm your AI assistant. What can I help you with today?`,
    voice: {
      provider: "11labs",
      voiceId: VOICES.female.rachel
    }
  };

  const headers = {
    'Authorization': `Bearer ${VAPI_CONFIG.AUTH_TOKEN}`,
    'Content-Type': 'application/json',
  };

  const data = {
    assistant: {
      firstMessage: config.firstMessage(customerName.trim()),
      model: {
        provider: "openai",
        model: "gpt-3.5-turbo",
        temperature: 0.8,
        systemPrompt: config.systemPrompt
      },
      voice: config.voice
    },
    phoneNumberId: VAPI_CONFIG.PHONE_NUMBER_ID,
    customer: {
      number: formattedNumber,
      name: customerName.trim()
    }
  };

  try {
    const response = await fetch(VAPI_CONFIG.API_URL, {
      method: 'POST',
      headers,
      body: JSON.stringify(data),
    });

    const responseData = await response.json();

    if (!response.ok) {
      return {
        status: 'failed',
        error: responseData.message || responseData.error || 'Failed to schedule call'
      };
    }

    return {
      status: 'scheduled',
      callId: responseData.callId,
      message: 'Call scheduled successfully'
    };
  } catch (error) {
    console.error('VAPI Error:', error);
    return {
      status: 'failed',
      error: error instanceof Error ? error.message : 'An unexpected error occurred'
    };
  }
}